<?php

declare(strict_types=1);

namespace Erlage\Photogram\Structures;

final class TableDataDock
{
    /**
     * @property TableDataDockContainer[]
     */
    public $containers = array();

    /**
     * @property string[]
     */
    private $registry = array();

    /**
     * get a container for table name. if not exists, will create it 
     */
    public function getContainer(string $tableName): TableDataDockContainer
    {
        if ($this -> containerExists($tableName))
        {
            return $this -> containers[$this -> containerIndex($tableName)];
        }

        $container = new TableDataDockContainer($tableName);

        $this -> registerContainer($tableName);

        $this -> containers[] = $container;

        return $container;
    }

    /**
     * Tells whether there are containers in dock that can be popped
     */
    public function canDispatch(): bool
    {
        return \count($this -> containers) > 0;
    }

    /**
     * Dispatching includes Poping a TableDataDockContainer from dock.
     * before dispatching container, ensure a success call to canDispatch
     */
    public function dispatchContainer(): TableDataDockContainer
    {
        /**
         * @var TableDataDockContainer
         */
        $container = \array_pop($this -> containers);

        $this -> unRegisterContainer($container -> getTableName());

        return $container;
    }

    /*
    |--------------------------------------------------------------------------
    | private helpers
    |--------------------------------------------------------------------------
    */

    private function containerExists(string $tableName): bool
    {
        return isset($this -> registry[$tableName]);
    }

    private function containerIndex(string $tableName): int
    {
        return $this -> registry[$tableName];
    }

    private function registerContainer(string $tableName): void
    {
        $this -> registry[$tableName] = \count($this -> containers);
    }

    private function unRegisterContainer(string $tableName): void
    {
        unset($this -> registry[$tableName]);
    }
}
